%%%-------------------------------------------------------------------
%%% File    : sipsocket_test.erl
%%% Author  : Fredrik Thulin <ft@it.su.se>
%%% Descrip.: Sipsocket test module, fakes network communication to
%%%           make it possible to test other modules.
%%%
%%% Created : 18 Jul 2005 by Fredrik Thulin <ft@it.su.se>
%%%-------------------------------------------------------------------
-module(sipsocket_test).
%%-compile(export_all).

-behaviour(sipsocket).

%%--------------------------------------------------------------------
%% External exports
%%--------------------------------------------------------------------
-export([
	 start_link/0,
	 send/5,
	 is_reliable_transport/1,
	 get_socket/1,
	 get_specific_socket/1,
	 get_raw_socket/1,
	 get_remote_peer/1,
	 close_socket/1
	]).

%%--------------------------------------------------------------------
%% Include files
%%--------------------------------------------------------------------
-include("sipsocket.hrl").


%%====================================================================
%% External functions
%%====================================================================

%%--------------------------------------------------------------------
%% Function: start_link()
%% Descrip.: Would've done some useful initialization if this was not
%%           merely a test module.
%% Returns : ignore
%%--------------------------------------------------------------------
start_link() ->
    ignore.


%%====================================================================
%% Interface functions
%%====================================================================

%%--------------------------------------------------------------------
%% Function: send(SipSocket, Proto, Host, Port, Message)
%%           SipSocket = sipsocket record()
%%           Proto     = atom(), yxa_test
%%           Host      = string()
%%           Port      = integer()
%%           Message   = term()
%% Descrip.: Fake sending Message to Host:Port. Return failure or
%%           success based on process dictionary.
%% Returns : ok              |
%%           {error, Reason}
%%           Reason = string()
%%--------------------------------------------------------------------
send(SipSocket, Proto, _Host, _Port, _Message)
  when is_record(SipSocket, sipsocket), SipSocket#sipsocket.proto /= Proto ->
    {error, "Protocol mismatch"};
send(SipSocket, yxa_test, Host, Port, Message) when is_record(SipSocket, sipsocket) ->
    Proto = SipSocket#sipsocket.proto,
    self() ! {sipsocket_test, send, {Proto, Host, Port}, Message},
    case autotest:is_unit_testing(?MODULE, {sipsocket_test, send_result}) of
	{true, {error, Reason}} ->
	    {error, Reason};
	false ->
	    ok
    end.

%%--------------------------------------------------------------------
%% Function: get_socket(Dst)
%%           Dst = sipdst record()
%% Descrip.: Return a fake socket or a term based on process dict.
%% Returns : sipsocket record() | term()
%%--------------------------------------------------------------------
get_socket(#sipdst{proto = yxa_test}) ->
    case autotest:is_unit_testing(?MODULE, {sipsocket_test, get_socket}) of
	{true, Res} ->
	    Res;
	false ->
	    #sipsocket{module = ?MODULE,
		       proto  = yxa_test,
		       pid    = self()
		      }
    end.

%%--------------------------------------------------------------------
%% Function: get_specific_socket(Id)
%%           Id = ob_id record()
%% Descrip.: Return a fake socket or a term based on process dict.
%% Returns : sipsocket record() | term()
%%--------------------------------------------------------------------
get_specific_socket(#ob_id{proto = yxa_test}) ->
    case autotest:is_unit_testing(?MODULE, {sipsocket_test, get_specific_socket}) of
	{true, Res} ->
	    Res;
	false ->
	    #sipsocket{module = ?MODULE,
		       proto  = yxa_test,
		       pid    = self()
		      }
    end.

%%--------------------------------------------------------------------
%% Function: get_raw_socket(SipSocket)
%%           SipSocket = sipsocket record()
%% Descrip.: Return a fake raw socket or a term based on process dict.
%% Returns : sipsocket record() | term()
%%--------------------------------------------------------------------
get_raw_socket(#sipsocket{proto = yxa_test}) ->
    case autotest:is_unit_testing(?MODULE, {sipsocket_test, get_raw_socket}) of
	{true, Res} ->
	    Res;
	false ->
	    {sipsocket_test, fake_raw_socket}
    end.

%%--------------------------------------------------------------------
%% Function: get_remote_peer(SipSocket)
%%           SipSocket = sipsocket record()
%% Descrip.: Return fake remote peer info based on process dictionary.
%% Returns : {ok, Proto, Addr, Port} | term()
%%           Proto = yxa_test
%%           Addr  = string(), "192.0.2.242"
%%           Port  = integer()
%%--------------------------------------------------------------------
get_remote_peer(#sipsocket{proto = yxa_test}) ->
    case autotest:is_unit_testing(?MODULE, {sipsocket_test, get_remote_peer}) of
	{true, Res} ->
	    Res;
	false ->
	    {ok, yxa_test, "192.0.2.242", sipsocket:get_listenport(yxa_test)}
    end.

%%--------------------------------------------------------------------
%% Function: is_reliable_transport(SipSocket)
%%           SipSocket = sipsocket record()
%% Descrip.: Fake response based on process dictionary.
%% Returns : true | false
%%--------------------------------------------------------------------
is_reliable_transport(#sipsocket{proto = yxa_test}) ->
    case autotest:is_unit_testing(?MODULE, {sipsocket_test, is_reliable_transport}) of
	{true, Res} ->
	    Res;
	false ->
	    false
    end.

%%--------------------------------------------------------------------
%% Function: close_socket(SipSocket)
%%           SipSocket = sipsocket record()
%% Descrip.: Fake response based on process dictionary.
%% Returns : ok              |
%%           {error, Reason}
%%           Reason = not_applicable | term()
%%--------------------------------------------------------------------
close_socket(#sipsocket{proto = yxa_test}) ->
    case autotest:is_unit_testing(?MODULE, {sipsocket_test, close_socket}) of
	{true, Res} ->
	    Res;
	false ->
	    ok
    end.

%%====================================================================
%% Internal functions
%%====================================================================

%%--------------------------------------------------------------------
%% Function:
%% Descrip.:
%% Returns :
%%--------------------------------------------------------------------
