/* Copyright (c) 2003, Stefan Petersen
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 
 * 3. Neither the name of the author nor the names of any contributors
 *    may be used to endorse or promote products derived from this 
 *    software without specific prior written permission.
 *   
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
 * ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 * 
 * This file is part of ccir tone detection
 *
 * $Id: tone.c,v 1.10 2004/01/17 11:32:58 spe Exp $
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>

#include <math.h>

#include "audio.h"

#define NUF_FREQS 15

struct tone_system {
    int samplerate;
    double N;  /* Could be int, but simplifies calculation further down */
    double N2; /* Could be int, but simplifies calculation further down */
    double hit_limit;
    const double *freqs;
};

const double _ccir_freqs[] = {1981.0, 1124.0, 1197.0, 1275.0, 1358.0, 
			      1446.0, 1540.0, 1640.0, 1747.0, 1860.0, 
			      2400.0, 930.0,	2247.0, 991.0, 2110.0};
const double _zvei_freqs[] = {2400.0, 1060.0, 1160.0, 1270.0, 1400.0,
			      1530.0, 1670.0, 1830.0, 2000.0, 2200.0,
			      2800.0, 810.0, 970.0, 886.0, 2600.0};
const double _eea_freqs[] = {1981.0, 1124.0, 1197.0, 1275.0, 1358.0,
			     1446.0, 1540.0, 1640.0, 1747.0, 1860.0,
			     1055.0, 930.0, 2247.0, 991.0, 2110.0};

const double _eia_freqs[] = {600.0, 741.0, 822.0, 1023.0, 1164.0, 
			     1305.0, 1446.0, 1587.0, 1728.0, 1869.0,
			     2151.0, 2433.0, 2010.0, 2292.0, 459.0};

const struct tone_system ts[] =
{{8000, 288.0, 299.0, 1450.0, _ccir_freqs},
 {8000, 280.0, 260.0, 1450.0, _zvei_freqs},
 {8000, 288.0, 299.0, 1450.0, _eea_freqs},
 {8000, 199.0, 227.0, 1450.0, _eia_freqs},
};


const char hextable[] = "0123456789ABCDEF";

int
main(int argc, char *argv[])
{
    int nuf_read, i, j, sample_cnt = 0, hit[NUF_FREQS], missed=0, curr_hit;
    int read_opt, curr_sys = 0;
    audio_setup_t a;
    unsigned char *abuf;
    double k, w, Q0[NUF_FREQS], Q1[NUF_FREQS], Q2[NUF_FREQS], 
	cosine[NUF_FREQS], sample;

    while ((read_opt = getopt(argc, argv, "czeih")) != -1) {
	switch (read_opt) {
	case 'c':
	    curr_sys = 0;
	    break;
	case 'z':
	    curr_sys = 1;
	    break;
	case 'e':
	    curr_sys = 2;
	    break;
	case 'i':
	    curr_sys = 3;
	    break;
	case 'h':
	    printf("%s [c|z|e|i] where c=ccir, z=zvei, e=eea, i=eia. ccir is default.\n", argv[0]);
	    exit(0);
	case '?':
	default:
	    fprintf(stderr, "Unknown option\n");
	    exit(0);
	}
    }

    a.channels = 1;
    a.speed = ts[curr_sys].samplerate;

    if (audio_init(AUDIO_DEV, &a) == -1) {
	fprintf(stderr, "Didn't manage to setup audio. Exiting!\n");
	exit(1);
    }

    abuf = (unsigned char *)malloc(a.blk_size*sizeof(unsigned char));
    if (abuf == NULL) {
	fprintf(stderr, "Failed to allocate audio buffer. Exiting!");
	exit(1);
    }

    for (i = 0; i < NUF_FREQS; i++) {
	/* In lack of an round() operator */
	if (modf((ts[curr_sys].N * ts[curr_sys].freqs[i])/(double)a.speed, &k) > 0.5)
	    k += 1.0;
	w = (2.0*M_PI*k)/ts[curr_sys].N;
	cosine[i] = 2.0*cos(w);
/*  	printf("F:%f k:%f w:%f cosine:%f\n", freqs[i], k, w, cosine[i]); */
    }

    memset(hit, 0, NUF_FREQS*sizeof(int));
    nuf_read = read(a.fd, abuf, a.blk_size);
    while (1) {
	memset(Q0, 0, NUF_FREQS*sizeof(double));
	memset(Q1, 0, NUF_FREQS*sizeof(double));
	memset(Q2, 0, NUF_FREQS*sizeof(double));
	
	/* 
	 * Feedback loop
	 */
	for (i = 0; i < ts[curr_sys].N; i++) {
	    
	    if (sample_cnt == nuf_read) {
		nuf_read = read(a.fd, abuf, a.blk_size);
		sample_cnt = 0;
	    }
	    sample = ((double)abuf[sample_cnt++] - 128.0) / 128.0;
	    
	    /* All frequencies */
	    for (j = 0; j < NUF_FREQS; j++) {
		Q2[j] = Q1[j];
		Q1[j] = Q0[j];
		Q0[j] = cosine[j] * Q1[j] - Q2[j] + sample;
	    }
	}

	/* 
	 * Feed forward
	 */
	for (j = 0; j < NUF_FREQS; j++) {
	    curr_hit = 0;
	    if (Q0[j]*Q0[j] + Q1[j]*Q1[j] -  Q0[j]*Q1[j]*cosine[j] > 
		ts[curr_sys].hit_limit) {
		hit[j] += 1;
		curr_hit += 1;
	    } else {
		hit[j] = 0;
	    }
	    if (hit[j] == 2) {
		missed = 0;
		printf("%c", hextable[j]);
		fflush(stdout);
	    } else missed++;
	    if (hit[j] == 7) printf("L");
	    if ((missed == 10*NUF_FREQS) && (curr_hit == 0)) {
		printf("\n");
		fflush(stdout);
	    }
	}
    }

    free(abuf);
    close(a.fd);

    return 0;
}
