; DISPLAY.INC
; LCD display routines for 16C84

; (C) 1997 Stefan Petersen, petersen@stacken.kth.se

;Following macros are implemented for easy use
;DEFSTR:	Defines a string for use with PRTSTR
;BL_ON	
;BL_OFF
;DISPCLR
;LINE1
;LINE2
;LCDINIT
;PRTSTR

;Following subroutines to call are implemented
;SENDI
;SENDD

; Setups for codegeneration
#define BACKLIGHTER
;#define LOCAL_WAIT


; LCD display pin and port assignment

LCDDATA		equ	PORTA	; i/o RA0-RA3 = 4-bit LCD-data bus
LCDCTRL		equ	PORTB	;   0 Control signals to LCD
DISP_E          equ     1       ;   o RB1 = Display (R/!W) Enable
DISP_RW         equ     2       ;   o RB2 = Display READ/!WRITE
DISP_RS         equ     3       ;   o RB3 = Display Registerselect
DISP_BL		equ	4	;   o RA4 = Backlighter on/off

BF_FLAG		equ	01h	; Display Busy Flag
I_FLAG		equ	02h	; Instruction Flag to indicate Instr to be sent
BL_FLAG		equ	03h	; Backlighter Flag backlighter off/on

; PIC16C84 RAM Register Assignments
	cblock
		LCDINFO		; Information storage byte
		DISPDATA	; Next Data to send to display
		STRPTR		; General loopcounter
	endc
#ifdef LOCAL_WAIT
	cblock
		TIME		; Down counting of microseconds
	endc
#endif


DEFSTR	macro	STRING	; Define String Macro
	addwf	PCL,F		; Get data as computed goto
	dt	STRING		; Store string as PIC-table
	retlw	0		; End-Of-String
	endm

BL_ON	macro		; Turn Backlighter On Macro
	bsf	LCDINFO,BL_FLAG	; Set info bit
	bcf	LCDDATA,DISP_BL	; Actuate command
	endm

BL_OFF	macro		; Turn Backlighter Off Macro
	bcf	LCDINFO,BL_FLAG	; Unset info bit
	bsf	LCDDATA,DISP_BL	; Actuate command
	endm

PORTALO	macro
	bsf     STATUS,RP0      ; Set page 1 for getting access to TRIS
	movlw   b'11100000'     ; Set Port A Tristate latches
	movwf   TRISA           ; Store in port a tristate register
	bcf     STATUS,RP0      ; Change back to page 0
	endm

PORTAHI	macro
	bsf     STATUS,RP0      ; Set page 1 for getting access to TRIS
	movlw   b'11101111'     ; Set Port A Tristate latches
	movwf   TRISA           ; Store in port A tristate register
	bcf     STATUS,RP0      ; Change back to page 0
	endm

CLKDISP	macro
	bsf	LCDCTRL,DISP_E	; Raise Enable pin on display
	bcf	LCDCTRL,DISP_E	; Lower Enable pin on display
	endm


DISPCLR	macro
	movlw   b'00000001'     ; Display Clear
	call    SENDI           ; Send instruction to display
	endm

LINE1	macro
	movlw	b'10000000'	; Set LCD address to first position
	call	SENDI
	endm

LINE2	macro
	movlw	b'11000000'	; Set LCD address to next line
	call	SENDI
	endm

LCDINIT	macro
	PORTALO			; Set port A as output

	movlw   0xF0            ; Make an inital wait for display in 15 ms
	call    LCDWAIT		; Call wait routine
	movlw   b'00000010'     ; Set 4-bit interface to to display
	movwf	LCDCTRL		; Send it to port A
	CLKDISP		        ; Clock out to display

	movlw   d'5'            ; Wait 5 ms so display understands 4-bit
	call    LCDWAIT		; Call wait routine

	movlw   b'00101000'     ; Set 4-bit interface, 2 lines & normal font
	call    SENDI           ; Send instruction to display
	movlw   b'00101000'     ; Set 4-bit interface, 2 lines & normal font
	call    SENDI           ; Send instruction to display
	movlw   b'00001000'     ; Display OFF
	call    SENDI           ; Send instruction to display
	DISPCLR
	movlw   b'00001100'     ; Display ON
	call    SENDI           ; Send instruction
	movlw   b'00000110'     ; Entry mode set to increment/no shift
	call    SENDI           ; Send instruction to display
	endm


PRTSTR	macro	STRING
	local	PRINTLOOP, ENDPRINTLOOP

	movlw   HIGH STRING		; Get higher bits of MESSAGES address
	movwf   PCLATH          ; Make high address OK for computed goto

	clrf	STRPTR		; Clr stringpointer
PRINTLOOP:
	movf	STRPTR,W	; Get stringpointer to W
	call	STRING		; Get letter from string based in index in W
	addlw	0		; Set status bits according to returned value
	btfsc	STATUS,Z	; Was it a zero (End-Of-String)?
	goto	ENDPRINTLOOP	; If yes quit this loop
	call	SENDD		; Else send character to display
	incf	STRPTR,F	; Increase stringpointer to next letter
	goto	PRINTLOOP 	; Make the loop
ENDPRINTLOOP:

	endm


; DISPLAY ROUTINES
SENDI:
	bsf	LCDINFO,I_FLAG	; Next data is an instruction
SENDD:
	movwf   DISPDATA        ; Store send-data for later use
	PORTAHI			; Put PortA tristate

	bcf	LCDCTRL,DISP_RS	; RS=0 => Instruction
	bsf	LCDCTRL,DISP_RW	; RW=1 => Read
DISPBUSY:
	bsf	LCDCTRL,DISP_E	; Display puts data on LCDDATA
	bcf     LCDINFO,BF_FLAG	; Busy flag is probably = 0
	btfsc   PORTA,3         ; Read busy flag from display
	bsf     LCDINFO,BF_FLAG	; Busy flag was = 1
	bcf	LCDCTRL,DISP_E	; Lower display enable port
	CLKDISP			; Clock out the last and uninteresting nibble
	btfsc   LCDINFO,BF_FLAG ; Check if display is ready
	goto    DISPBUSY        ; The display is not ready

	PORTALO			; Put portA in a low impedance state
	bsf	LCDCTRL,DISP_RS	; RS=1 => Data
	bcf	LCDCTRL,DISP_RW	; RW=0 => Write
	btfsc   LCDINFO,I_FLAG	; Check if SENDData?
	bcf	LCDCTRL,DISP_RS	; No, RS=0 => Instruction
	swapf   DISPDATA,W      ; Put first nibble to send in W-low
	movwf   LCDDATA		; Put first nibble to portA
	CLKDISP			; Clock out the first nibble to display
	movf    DISPDATA,W      ; Put last nibble to send in W
	movwf	LCDDATA		; Put last nibble on portA
	CLKDISP			; Clock out the last nibble to display
#ifdef BACKLIGHTER
	bsf	LCDDATA,DISP_BL	; Turn Backlighter off
	btfsc	LCDINFO,BL_FLAG	; Should backlighter be on or off?
	bcf	LCDDATA,DISP_BL	; Apperently it should be on
#endif
	bcf     LCDINFO,I_FLAG	; Assume next send is SENDD
	return

#ifdef LOCAL_WAIT
LCDWAIT:
	clrf	TIME		; Clear time downcounter
WAITLOOP:
	nop			; Extend timerloop to 256
	decfsz	TIME,F		; Decrement timer
	goto	WAITLOOP	; and loop if not down to zero again
	movwf	TIME		; W contains outer loop value
	decfsz	TIME,W		; Decrement
	goto	LCDWAIT
	return
#endif
