/* Functions for transmission of POCSAG codes *
 * (C) Stefan Petersen 1995                   *
 */

#include <stdio.h>
#include <malloc.h>
#include "pocsag.h"

#ifndef NULL
#define NULL ((void *)0)
#endif

/* Standard CodeWord's */
/*const CodeWord SYNCH = 0x7CD215D8;
const CodeWord IDLE  = 0x7A89C197; */


/* Internal functions */
/* Returns nuf CodeWord packed down to. Only the 20 LSBits are used in *
 * CodeWord.                                                           */
int pack_string(CodeWord *packed, char *unpacked, int size_up, MsgType type);

/* Creates a new Batch, fills in SC, nextptr and fills frames with IDLE *
 * Returns the pointer to the newly allocated Batch.                    */
Batch *create_batch(void);

/* Calculates the bch-crc-checksum and even parity on th CodeWord entered *
 * Returns CodeWord with checksum and paritybit                           */
CodeWord calc_bch_and_parity(CodeWord cw);


Batch *make_fill_batch(CodeWord RIC, 
		       char *data, int size_of_data,
		       MsgType type)
{
  Batch* now = NULL;
  Batch* top = NULL;
  int frame_addr = 0;
  int i,j,k = 0;
  CodeWord packed_text[100];
  int packed_length = 0;

  /* Create first batch with address and so on */
  now = create_batch();
  top = now;  /* Pointer to first batch in the row */

  /* Fill in addressframe with address and Function Bits */
  frame_addr = RIC & 7; /* Frameaddress == The three LSBits */
  now->frame[frame_addr][0] = 0;
  now->frame[frame_addr][0] = (RIC >> 3) << 13;

  /* Set functionbits according to what type of data */
  switch (type)
    {
    case Tone:    /* Tonecoding just sets functionbits */
      printf("Tone:\n");
      now->frame[frame_addr][0] &= 0xFFFFE700;
      switch(data[0])
	{
	case 0:
	  break;
	case 1:
	  now->frame[frame_addr][0] |= 0x00000800;
	  break;
	case 2:
	  now->frame[frame_addr][0] |= 0x00001000;
	  break;
	case 3:
	  now->frame[frame_addr][0] |= 0x00001800;
	  break;
	default:
	  fprintf(stderr, "Illegal Tone requested\n");
	  exit(1);
	}
    case Numerical:
      /* Don't set any bits */
      break;
    case Alpha:
      now->frame[frame_addr][0] |= 0x00001800;
      break;
    default:
      fprintf(stderr,"Unknown format in make_fill_batch\n");
      exit(1);
    }

  /* Calculate checksumm and parity for address CodeWord */
  now->frame[frame_addr][0] = calc_bch_and_parity( now->frame[frame_addr][0] );

  /* Start filling in data in frames */
  /* Warning very tricky part with i, j and k */
  if (type != Tone){  /* But only if it's alpha or numerical */
    packed_length = pack_string(packed_text, data, size_of_data, type);
    for(i = 0, k = 1, j = frame_addr; i < packed_length; k=0, j++){
      for(; k <= 1; k++, i++){
	if (i == packed_length) break;
	if (j == 8){             /* If we filled a whole batch, */
	  j = 0; k = 0;          /* start over with a new batch */
	  now->next = create_batch();
	  now = now->next;
	}
	now->frame[j][k] = packed_text[i];
	now->frame[j][k] <<= 11; /* Make room for CRC and parity */
	now->frame[j][k] |= 0x80000000; /* Set msg CW-bit */
	now->frame[j][k] = calc_bch_and_parity( now->frame[j][k] ); 
      }
    }
  }
  return top;
}

void kill_batch(Batch *batch)
{
  if (batch){ /* Pointing to something */
    kill_batch(batch->next);
  }else{
    return;
  }
  free(batch);
  return;
}

CodeWord calc_bch_and_parity(CodeWord cw)
{
  int bit=0;
  int local_cw = 0;
  int parity = 0;

  local_cw=cw;  /* bch */
  for(bit=1; bit<=21; bit++, cw <<= 1)
    if (cw & 0x80000000) cw ^= 0xED200000;
  local_cw |= (cw >> 21);

  cw =local_cw;  /* parity */
  for(bit=1; bit<=32; bit++, cw<<=1)
    if (cw & 0x80000000) parity++;

  return (parity%2) ? local_cw+1 : local_cw;

}

/* Returns nuf CodeWord packed down to. Only the 20 LSBits are used in *
 * CodeWord.                                                           */
int pack_string(CodeWord *packed, char *unpacked, int size_up, MsgType type)
{
  int real_size = 0;
  int upi = 0, pi = 0;

  switch (type)
    {
    case Tone:
      fprintf(stderr, "Severe Error; Trying to pack Tone data\n");
      exit(1);
    case Numerical: 
      real_size = ((size_up/5)+1)*5; /* One CodeWord = 5 BCD's */
      for(upi=0, pi=0; upi<real_size; upi++, pi=(upi%5?pi:pi+1)){
	if (upi<size_up)
	  packed[pi] = (packed[pi]<<4) + (unpacked[upi] & 0x0F);
	else
	  packed[pi] = (packed[pi]<<4);
      }
      return real_size/5;
    case Alpha:
      fprintf(stderr, "Boring Error; I'm not intrested to invent an ");
      fprintf(stderr, "algorithm for pack ascii. Quiting!\n");
      exit(1);
      break;
    default:
      fprintf(stderr, "Severe Error; Illegal MsgType to pack_string\n");
      exit(1);
    }
  return 0;
}

/* Creates a new Batch, fills in SC, nextptr and fills frames with IDLE *
 * Returns the pointer to the newly allocated Batch.                    */
Batch *create_batch(void)
{
  Batch *temp;
  int i,j;

  if ((temp = malloc(sizeof(Batch))) == NULL){
    printf("Could not mallocate space for a new batch.\n");
    exit(1);
  }
  temp->sc   = SYNCH;
  temp->next = NULL;

  /* Fill frames with IDLE */
  for (i=0; i<8; i++){
    for (j=0; j<2; j++){
      temp->frame[i][j] = IDLE;
    }
  }
  return temp;
}
