/* Copyright (c) 2003, Stefan Petersen
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 
 * 3. Neither the name of the author nor the names of any contributors
 *    may be used to endorse or promote products derived from this 
 *    software without specific prior written permission.
 *   
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
 * ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 * 
 * This file is part of ccir tone detection
 *
 * $Id: tone.c,v 1.7 2003/10/02 20:57:29 spe Exp $
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>

#include <math.h>

#include "audio.h"

#define NUF_FREQS 15
#define N 288
#define SAMPLERATE 8000
#define HIT_LIMIT 1450.0 

const double freqs[NUF_FREQS] = {1981.0, 1124.0, 1197.0, 1275.0, 1358.0, 
				 1446.0, 1540.0, 1640.0, 1747.0, 1860.0, 
				 2400.0, 930.0,	2247.0, 991.0, 2110.0};

int
main(int argc, char *argv[])
{
    int nuf_read, i, j, sample_cnt = 0, hit[NUF_FREQS], missed=0, curr_hit;
    audio_setup_t a;
    unsigned char *abuf;
    double k, w, Q0[NUF_FREQS], Q1[NUF_FREQS], Q2[NUF_FREQS], 
	cosine[NUF_FREQS], sample;

    a.channels = 1;
    a.speed = SAMPLERATE;

    if (audio_init(AUDIO_DEV, &a) == -1) {
	fprintf(stderr, "Didn't manage to setup audio. Exiting!\n");
	exit(1);
    }

    abuf = (unsigned char *)malloc(a.blk_size*sizeof(unsigned char));
    if (abuf == NULL) {
	fprintf(stderr, "Failed to allocate audio buffer. Exiting!");
	exit(1);
    }

    for (i = 0; i < sizeof(freqs)/sizeof(freqs[0]); i++) {
	/* In lack of an round() operator */
	if (modf(((double)N * freqs[i])/(double)a.speed, &k) > 0.5)
	    k += 1.0;
	w = (2.0*M_PI*k)/(double)N;
	cosine[i] = 2.0*cos(w);
/*  	printf("F:%f k:%f w:%f cosine:%f\n", freqs[i], k, w, cosine[i]); */
    }

    memset(hit, 0, NUF_FREQS*sizeof(int));
    nuf_read = read(a.fd, abuf, a.blk_size);
    while (1) {
	memset(Q0, 0, NUF_FREQS*sizeof(double));
	memset(Q1, 0, NUF_FREQS*sizeof(double));
	memset(Q2, 0, NUF_FREQS*sizeof(double));
	
	/* 
	 * Feedback loop
	 */
	for (i = 0; i < N; i++) {
	    
	    if (sample_cnt == nuf_read) {
		nuf_read = read(a.fd, abuf, a.blk_size);
		sample_cnt = 0;
	    }
	    sample = ((double)abuf[sample_cnt++] - 128.0) / 128.0;
	    
	    /* All frequencies */
	    for (j = 0; j < sizeof(freqs)/sizeof(freqs[0]); j++) {
		Q2[j] = Q1[j];
		Q1[j] = Q0[j];
		Q0[j] = cosine[j] * Q1[j] - Q2[j] + sample;
	    }
	}

	/* 
	 * Feed forward
	 */
	for (j = 0; j < sizeof(freqs)/sizeof(freqs[0]); j++) {
	    curr_hit = 0;
	    if (Q0[j]*Q0[j] + Q1[j]*Q1[j] -  Q0[j]*Q1[j]*cosine[j] > HIT_LIMIT) {
		hit[j] += 1;
		curr_hit = 1;
	    } else {
		hit[j] = 0;
	    }
	    if (hit[j] == 2) {
		missed = 0;
		if (j > 9)
		    printf("%c", j+'A'-10);
		else
		    printf("%d", j);
		fflush(stdout);
	    } else missed++;
	    if (hit[j] == 7) printf("L");
	    if ((missed == 10*NUF_FREQS) && !curr_hit) {
		printf("\n");
		fflush(stdout);
	    }
	}
    }

    free(abuf);
    close(a.fd);

    return 0;
}
