%%% This is an example of how an antenna pattern could be
%%% created using a triangle as the S function
%%%
%%% Written by Johan Skatt, Royal Institute of Technology 1997

clear,clf

%%%	Here's were you give the specifications !

	% Makes theta to zero at broadside
	thetar=pi/2;

	% The limits for the beam
	theta0=110*pi/180;
	theta1=150*pi/180;

	% N1 are the number of zeros in region I (the beam)
	N1=10;
	% N2 are the number of zeros in region II (the sidelobes)
	N2=30;

	n=1:N1+N2;

%%% Create wanted g 

	gg=[];
	
	% -20 dB in first half of region II, using 25 zeros
	gg=[gg -20*ones(1,25)];

	% -30 dB in second half of region II, using 5 zeros
	gg=[gg -30*ones(1,5)];

	% 0.5dB ripple in region I, using 10 zeros
	for k=1:21
 		if rem(k,2)==0 gg=[gg -0.5];
  	else gg=[gg 0.5]; end
	end

%%% End of specification !

%%% Some handy constants

	fi=-pi:0.01:pi;
	K=180/pi;

%%% Some initializing
	

	% Generate some nice starting values for a, b, C1 and C2 
	b=(2*n/42-1)*pi;
	a=0.02*ones(1,N1);
	C1=0;
	C2=0;
	% Assemble a, b and C1 into the iteration vector x
	x=[a b C1];

%%% Here is the main loop !

for n=1:5
	% Find psi at maximas/minimas
	psii=Newton(x,N1,N2);
	psi2=psii(:,1:N2);
	psi1=psii(:,N2+1:2*N1+N2+1);

	% Find the rotate angle psir
	psi0=psii(length(psii));
	psir=psi0-pi*cos(theta0);

	% Find the values for G-S at psi1, S is the triangle function

	S=[0 1 2 3 4 5 6 7 8 9 10 9 8 7 6 5 4 3 2 1 0]-10;
 	g1=G(x,psi1,N1,N2)-S;

	% Find the values for G at psi2 
	g2=G(x,psi2,N1,N2);

	% Assemble into the gtop vector
	gtop=[g2 g1];

	% Calculate Jacobian
	A=Jacobi(x,psii,N1,N2);

	% Calculate deltax
	deltax=A\(gg'-gtop');
	deltax=deltax';
	x=x+deltax;
end

%%% End of main loop !

%%% Plot the wanted beam in the psi-plane
	plot(K*fi,G(x,fi,N1,N2)), axis([-180 180 -35 20]);
	%title('Beamform in the psi-plane');
	print -deps abs.eps
	pause

%%% Plot the zeros in the w-plane
	a1=x(:,1:N1);
	a2=zeros(1,N2);
	b2=x(:,N1+1:N1+N2);
	b1=x(:,N1+N2+1:2*N1+N2);
	a=[a2 a1];
	b=[b2 b1];
	polar(b,exp(a),'ro');
	title('Unoptimized zeros in the w-plane');
	pause
	
%%% Calculate the optimum exitation, by means of minimazing 
%%% the relationship Imax/Imin.
	disp('Trying to optimize the result');
	% Generate the binary matrix with an external program (N1=10)
	disp('Generating the matrix');
	!binmat 10 binary.m

	[ourI]=bestI(x,N1,N2);

	[optimalI,optimalIndex]=min(ourI(:,1));
	x(1:N1) = x(1:N1).*ourI(optimalIndex, 2:N1+1);

%%% Plot the optimized zeros in the w-plane
	a1=x(:,1:N1);
	a2=zeros(1,N2);
	b2=x(:,N1+1:N1+N2);
	b1=x(:,N1+N2+1:2*N1+N2);
	a=[a2 a1];
	b=[b2 b1];
	polar(b,exp(a),'go');
	title('Optimized zeros in the w-plane');
