;;; gEDA - GNU Electronic Design Automation
;;; gnetlist - GNU Netlist
;;; Copyright (C) 1998 Ales V. Hvezda
;;;
;;; This program is free software; you can redistribute it and/or modify
;;; it under the terms of the GNU General Public License as published by
;;; the Free Software Foundation; either version 2 of the License, or
;;; (at your option) any later version.
;;;
;;; This program is distributed in the hope that it will be useful,
;;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;;; GNU General Public License for more details.
;;;
;;; You should have received a copy of the GNU General Public License
;;; along with this program; if not, write to the Free Software
;;; Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

;; --------------------------------------------------------------------------
;;
;; TANGO netlist backend written by Nuno Sucena starts here
;; Heavily modified by Stefan Petersen for strictly private purpose



;; If list of connections contains a pin namned after power
;; (eg "GND" or "+5V") then it's considered a power net
(define (power-pin? power connections)
  (cond ((null? connections)
	 #f)
	 ((string=? power (cadar connections)) ; power == pin?
	  #t)
	 (else
	  (power-pin? power (cdr connections)))))

(define GND-net '())
(define +5V-net '())


; split string at current split-char and returns
; a pair with substrings
;; string-index is a guile function

(define (string-split the-string split-char)
  (let ((split-index (string-index the-string split-char)))
    (if split-index
	(cons (substring the-string 0 split-index)
	      (substring the-string (+ split-index 1) 
			 (string-length the-string)))
	#f)))

(define (split-to-list attribute)
  (let ((the-list (string-split attribute #\;)))
    (cond ((not the-list)
	   (list (string-split attribute #\=)))
	  (else
	   (cons (string-split (car the-list) #\=)
		 (split-to-list (cdr the-list)))))))
	   

(define (add-to-net! net-name package pin)
  (let ((complete-name (list package pin)))
    (case (string->symbol net-name)
      ((GND) (set! GND-net (cons complete-name GND-net)))
      ((+5V) (set! +5V-net (cons complete-name +5V-net)))
      (else
       (display "Undefined net")))))


; Goes through all packages to see if there is any attribute
; powernet added to that package. If so that attribute is
; evaluated for the powernet and is added to resp list.
(define (examine-powernet-attribute)
  (for-each 
   (lambda (package)
     (let ((attribute (gnetlist:get-package-attribute package "powernet")))
       (if (not (string=? "unknown" attribute))
	   (for-each 
	    (lambda (connection)
	      (add-to-net! (cdr connection) package (car connection)))
	    (split-to-list attribute)))))
   (gnetlist:get-packages "placeholder")))


;; Cleans out all components having a pin called pin-name in the
;; list whivh defines this net.
(define (clean-powernet pin-name net)
  (if (null? net)
      '()
      (let ((package (caar net))
	    (pin (cadar net)))
	(if (string=? pin-name pin)
	    (clean-powernet pin-name (cdr net))
	    (cons (car net) (clean-powernet pin-name (cdr net)))))))


;;
;; ORIGINAL CODE START, THOUGH SLIGHTLY MOFIFIED
;;
    

;;
;; Given a uref, returns the device attribute value (for tango-netlist)
;;
(define tango:get-device
  (lambda (package)
    (gnetlist:get-package-attribute package "device")))

;;
;; Given a uref, returns the pattern attribute value (PATTERN if not defined)
;;
(define tango:get-pattern
  (lambda (package)
    ;; CHANGED define => let
    (let ((pattern (gnetlist:get-package-attribute package "pattern")))
      (if (string=? "unknown" pattern)
	  "PATTERN"
	  pattern))))

;;
;; Given a uref, returns the value attribute (empty if not defined)
;;
(define (tango:get-value package)
;; CHANGED
  (let ((value (gnetlist:get-package-attribute package "value")))
    (if (string=? "unknown" value)
	""
	value)))
 
;;
;; Top level header
;;
(define tango:write-top-header
  (lambda (p)
    (display "START header" p) 
    (newline p)
    (newline p)
    (display "TANGO netlist for gnetlist" p)
    (newline p)
    (display "TANGO gnetlist backend written by Nuno Sucena" port)
    (newline p)
    (display "END header" p)
    (newline p)
    (newline p)))

;;
;; Top level component writing 
;;
(define tango:components
  (lambda (port ls)
    (if (not (null? ls))
	(let* ((package (car ls))
	       (device (tango:get-device package)))
	  (if (and (not (string=? device "GND"))
		   (not (string=? device "VDD")))
	      (begin
		(display "[" port)
		(newline port)
		(display package port)
		(newline port)
		(display (tango:get-pattern package) port)
		(newline port)
		(display device port)
		(newline port)
		(display (tango:get-value package) port)
		(newline port)
		(newline port)
		(display "]" port)
		(newline port)))
	  (tango:components port (cdr ls))))))

;;
;; Display the individual net connections
;;
(define tango:display-connections
  (lambda (nets port)
    (if (not (null? nets))
	(begin
	  (display (car (car nets)) port)
	  (display "-" port) 
	  (display (car (cdr (car nets))) port)
;	  (newline port)
	  (if (not (null? (cdr nets)))
	      (newline port))
	  (tango:display-connections (cdr nets) port)))))


;;
;; Properly format the name of the net and the actual net connections
;;
(define tango:display-name-nets
  (lambda (port net)
    (let ((netname (car net)) 
	  (nets (cdr net)))
      (cond ((power-pin? "GND" nets) ;; GND net?
	     ;(display "GND:")
	     ;(display nets)
	     ;(newline)
	     (set! GND-net (append GND-net nets)))
	    ((power-pin? "+5V" nets) ;; +5V net?
	     ;(display "+5V:")
	     ;(display nets)
	     ;(newline)
	     (set! +5V-net (append +5V-net nets)))
	    ((and 
	      (not (string=? "duplicate" netname))
	      (not (string=? "unconnected_pin" netname))) ;;CHANGED
	     (display "(" port)
	     (newline port)
	     (display netname port)
	     (newline port)
	     (tango:display-connections nets port)
	     (newline port)
	     (display ")" port)
	     (newline port))))))

;;
;; Write out a net associated with a particular package and pin
;;
(define tango:write-net
  (lambda (port package pins)
    (if (not (null? pins))
	(let ((pin (car pins)))
	  (tango:display-name-nets port (gnetlist:get-nets package pin))
	     (tango:write-net port package (cdr pins))))))


;;
;; Top level function to write out nets associated with a particular component
;;
(define tango:nets
  (lambda (port ls)
    (if (not (null? ls)) 
	(let* ((package (car ls)) ;;CHANGED
	       (package-pins (gnetlist:get-pins package)))
	  (tango:write-net port package package-pins)
	  (tango:nets port (cdr ls))))))


;;; Highest level function
;;; Write tango netlist format
;;;
(define tango
  (lambda (output-filename)
    (let ((port (open-output-file output-filename))
	  (packages (gnetlist:get-packages "placeholder")))
      (display "WARNING: gnetlist still has some serious bugs -- bogus netlists are possible!") 
	 (newline)
	 (gnetlist:set-netlist-mode "TANGO")
	 (tango:components port packages)
	 (tango:nets port packages)

	 (examine-powernet-attribute)

	 (set! GND-net (clean-powernet "GND" GND-net))
	 (set! +5V-net (clean-powernet "+5V" +5V-net))

	 (display "+5V = ")
	 (display +5V-net)
	 (newline)
	 (display "GND = ")
	 (display GND-net)
	 (newline)

; Store powernets
	 (display "(" port)
	 (newline port)
	 (display "GND" port)
	 (newline port)
	 (tango:display-connections GND-net port)
	 (newline port)
	 (display ")" port)
	 (newline port)

	 (display "(" port)
	 (newline port)
	 (display "+5V" port)
	 (newline port)
	 (tango:display-connections +5V-net port)
	 (newline port)
	 (display ")" port)
	 (newline port)
	 
	 (close-output-port port))))

;;
;; TANGO netlist backend written by Nuno Sucena ends here
;;
;; --------------------------------------------------------------------------

