/* Copyright (c) 2003, Stefan Petersen
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 
 * 3. Neither the name of the author nor the names of any contributors
 *    may be used to endorse or promote products derived from this 
 *    software without specific prior written permission.
 *   
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
 * ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 * 
 * This file is part of ape, Another Packet Engine
 *
 * $Id: corr.c,v 1.1.1.1 2003/09/10 18:35:54 spe Exp $
 *
 */

#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <string.h>

#include <math.h>
#include <sndfile.h>

#include "corr.h"
#include "audio.h"
#include "mpt1327.h"

#define SAMPLE_BUF_SIZE 4096

mpt1327_t mpt1327_state;

/* 
 * Early-late gate PLL symbol recovery
 */
static int
sym_recovery_1200(FILE *fd, u_int16_t corr, unsigned char nuf_ones, int samplespersym)
{
    int diff = 0;

    /* 
     * Enough ones in a row gives a ones. The next algorithm is to keep
     * correlated ones together to be able to here determine if they are 
     * ones or not.
     */
    if (nuf_ones > 4) { /* 1 */
	mpt1327(1, &mpt1327_state);
    } else { /* 0 */
	mpt1327(0, &mpt1327_state);
    }

    /* 
     * This algorithm is basically a version of early-late gate.
     * The goal is to keep as much ones centered in the vector.
     * By going from the middle and out we can find where the ones
     * are offsetted and then compensate for it.
     * We have to have enough 1's in the vector to safely guess were the
     * middle is.
     */
    if (nuf_ones > 4) {
	if ((corr & 0x0028) != 0x0028) {
	    if (corr & 0x0008)
		diff = 3;
	    else
		diff = -3;
	} else if ((corr & 0x0044) != 0x0044) {
	    if (corr & 0x0004)
		diff = 2;
	    else
		diff = -2;
	} else if ((corr & 0x0082) != 0x0082) {
	    if (corr & 0x0002)
		diff = 1;
	    else
		diff = -1;
	} else if ((corr & 0x0101) != 0x0101) {
	    diff = 0;
	} else {
	    diff = 0;
	}
    }

    return(samplespersym + diff); 

} /* sym_recovery_1200 */

    
void
corr(FILE *datafd, comm_info_t *comm_info)
{
    int samplespersym, i, nuf_read, curr_sample = 0;
    double *i0, *q0, *i1, *q1;
    double *corr_i0, *corr_q0, *corr_i1, *corr_q1;
    double sample;
    unsigned char *buffer;
    double sum[] = {0.0, 0.0, 0.0, 0.0};
    u_int16_t corr = 0, ones_calc_mask = 0;
    unsigned char nuf_ones = 0, pll = 0;
    audio_setup_t audio;
    double abs_sum;

    /*
     * Initialize audio source 
     */
    audio.channels = 1;
    audio.speed = 10800;
    audio_init("/dev/dsp", &audio);
    buffer = (unsigned char*)malloc(audio.blk_size);

    /*
     * Once in a run setups
     */
    samplespersym = (int)floor(audio.speed/comm_info->bps);

    i0 = (double *)malloc(samplespersym * comm_info->uneven * sizeof(double));
    q0 = (double *)malloc(samplespersym * comm_info->uneven * sizeof(double));
    i1 = (double *)malloc(samplespersym * comm_info->uneven * sizeof(double));
    q1 = (double *)malloc(samplespersym * comm_info->uneven * sizeof(double));
    
    corr_i0 = (double *)malloc(samplespersym * sizeof(double));
    corr_q0 = (double *)malloc(samplespersym * sizeof(double));
    corr_i1 = (double *)malloc(samplespersym * sizeof(double));
    corr_q1 = (double *)malloc(samplespersym * sizeof(double));

    memset((void *)corr_i0, 0, samplespersym * sizeof(double));
    memset((void *)corr_q0, 0, samplespersym * sizeof(double));
    memset((void *)corr_i1, 0, samplespersym * sizeof(double));
    memset((void *)corr_q1, 0, samplespersym * sizeof(double));

    for (i = 0; i < samplespersym * comm_info->uneven ; i++) {
	i0[i] = sin(2.0*M_PI*((double)i/(double)samplespersym) *
		    ((double)comm_info->freq0/(double)comm_info->bps));
	q0[i] = cos(2.0*M_PI*((double)i/(double)samplespersym) *
		    ((double)comm_info->freq0/(double)comm_info->bps));
	i1[i] = sin(2.0*M_PI*((double)i/(double)samplespersym) *
		    ((double)comm_info->freq1/(double)comm_info->bps));
	q1[i] = cos(2.0*M_PI*((double)i/(double)samplespersym) *
		    ((double)comm_info->freq1/(double)comm_info->bps));
    }

    /*
     * Run correlation, one loop per sample
     */
    pll = samplespersym;
    ones_calc_mask = (1<<(samplespersym - 1));

    while (1) {
	nuf_read = read(audio.fd, buffer, audio.blk_size);
	for (i = 0; i < nuf_read; i++) {
	    sample = ((double)buffer[i]-128.0)/128.0;
	    
	    sum[0] -= corr_i0[curr_sample%samplespersym];
	    sum[1] -= corr_q0[curr_sample%samplespersym];
	    sum[2] -= corr_i1[curr_sample%samplespersym];
	    sum[3] -= corr_q1[curr_sample%samplespersym];
	    
	    corr_i0[curr_sample%samplespersym] = i0[curr_sample] * sample;
	    corr_q0[curr_sample%samplespersym] = q0[curr_sample] * sample;
	    corr_i1[curr_sample%samplespersym] = i1[curr_sample] * sample;
	    corr_q1[curr_sample%samplespersym] = q1[curr_sample] * sample;
	    
	    sum[0] += corr_i0[curr_sample%samplespersym];
	    sum[1] += corr_q0[curr_sample%samplespersym];
	    sum[2] += corr_i1[curr_sample%samplespersym];
	    sum[3] += corr_q1[curr_sample%samplespersym];
	    
	    curr_sample++;
	    curr_sample %= (samplespersym * comm_info->uneven);
	    
	    if (corr & ones_calc_mask)
		nuf_ones--;
	    
	    corr <<= 1;
	    abs_sum = fabs(sum[2]) + fabs(sum[3]) - fabs(sum[0]) - fabs(sum[1]);
	    if (abs_sum > 0.0){
		corr += 1;
		nuf_ones++;
/*  		printf("%4.2f\n", fabs(abs_sum)); */
	    }
	    
	    pll--;	
	    if (pll == 0) {
/*  		if (comm_info->bps == 1200) */
		    pll = sym_recovery_1200(datafd, corr, nuf_ones, samplespersym);
/*  		else if (comm_info->bps == 2400) */
/*  		    pll = sym_recovery_2400(datafd, corr, nuf_ones, samplespersym); */
/*  		else  */
/*  		    printf("Unknown bps\n"); */
	    }
	}
    }
    
    /*
     * Clean up
     */
    free(i0);
    free(q0);
    free(i1);
    free(q1);

    free(corr_i0);
    free(corr_q0);
    free(corr_i1);
    free(corr_q1);
    
    return;
} /* corr */
